/*
 * HAL API:
 * LCD driver
 *
 * NOTE: currently only writing to the LCD is used by this driver;
 *       i.e. the RW line is initialised low (0) and left at that,
 *       it is assumed that no part of the program changes that.
 *       The only exception is the _lcd_check_busy() routine which
 *       temporarily sets RW high (1) to read the busy flag.
 *
 * $Id: lcd.c 53 2008-10-21 19:03:44Z gerry $
 */

#include "defines.h"
#include <avr/io.h>
#include <compat/ina90.h>
#include "services.h"
#include "lcd.h"

/* ===== local utility functions ===== */

static void _strobe_E(void)
{
    _NOP();
    _NOP();
    LCD_CTRL_PORT |=  (_BV(LCD_E));      /* E 0 -> 1 start RD access */
    _NOP();
    _NOP();
    _NOP();
    _NOP();
    LCD_CTRL_PORT &= ~(_BV(LCD_E));      /* E 1 -> 0  end  RD access */
}

static void _lcd_check_busy(void)
{
char busy = 1;

    LCD_DATA_PORT |= LCD_DATA_INIT;          /* drive all data lines high */
    LCD_DATA_DDIR &= ~(_BV(LCD_BF));         /* set Busy Flag pin as input */
    LCD_CTRL_PORT &= ~(_BV(LCD_RS));         /* RS -> 0  instruction register */
    LCD_CTRL_PORT |=  (_BV(LCD_RW));         /* set R/W pin for reading */
    while (busy)
    {
                                    /* 1st strobe reads higher nibble D7-D4 */
        LCD_CTRL_PORT |=  (_BV(LCD_E));      /* E 0 -> 1 start RD access */
        _NOP();
        _NOP();
        busy = LCD_DATA_PINS & _BV(LCD_BF);  /* save value of the busy flag */
        LCD_CTRL_PORT &= ~(_BV(LCD_E));      /* E 1 -> 0  end  RD access */
        _NOP();
        _NOP();
        _strobe_E();                /* dummy to complete the read cycle */
    }
    LCD_DATA_DDIR |=  (_BV(LCD_BF));         /* set Busy Flag pin as output */
    LCD_CTRL_PORT &= ~(_BV(LCD_RW));         /* set R/W pin for writing */
}

static void _lcd_init_wr(char cmd)        /* this routine writes only the    */
{                                         /* lower nibble onto the 4-bit bus */
    LCD_CTRL_PORT &= ~(_BV(LCD_RS));      /* RS -> 0  instruction code input */
    LCD_DATA_PORT &= ~LCD_DATA_INIT;      /* set data bits only to zero */
    LCD_DATA_PORT |= (0x0F & (cmd));      /* next change only the data bits */
    _strobe_E();
}

static void _lcd_control_wr(char cmd)
{
    _lcd_check_busy();                    /* wait til device not busy */
    LCD_CTRL_PORT &= ~(_BV(LCD_RS));      /* RS -> 0  instruction code input */
    LCD_DATA_PORT &= ~LCD_DATA_INIT;      /* set data bits only to zero */
    LCD_DATA_PORT |= (0x0F & (cmd >> 4)); /* first set the higher nibble */
    _strobe_E();
    LCD_DATA_PORT &= ~LCD_DATA_INIT;      /* set data bits only to zero */
    LCD_DATA_PORT |= (0x0F & (cmd));      /* next set the lower nibble */
    _strobe_E();
}



/*
 * Initialize the LCD to 4-bit interface
 */
void lcd_init(void)
{
    /* initialize the uP interface towards the LCD */
    LCD_DATA_PORT |= LCD_DATA_INIT;
    LCD_DATA_DDIR |= LCD_DATA_OUTP;
    LCD_CTRL_PORT |= LCD_CTRL_INIT;
    LCD_CTRL_DDIR |= LCD_CTRL_OUTP;
    LCD_CTRL_PORT &= ~(_BV(LCD_E)|_BV(LCD_RS)|_BV(LCD_RW));
    /* Initialize the display */
    _delay_ms(30);       /* wait  30 ms */
    _lcd_init_wr(0x03);
    _delay_ms(5);        /* wait  5 ms */
    _lcd_init_wr(0x03);
    _delay_ms(1);        /* wait 1 ms */
    _lcd_init_wr(0x03);

    _lcd_check_busy();
    _lcd_init_wr(0x02);  /* set to 4 bit interface */

    _lcd_control_wr(LCD_SYSTEM_SET_2_LINES_5x7_CHAR);
    _lcd_control_wr(LCD_DISPLAY_OFF_CURSOR_OFF);
    _lcd_control_wr(LCD_CLEAR_DISPLAY);
    _lcd_control_wr(LCD_MODE_SET_INC_CURSOR);
    _lcd_control_wr(LCD_CURSOR_HOME);
    _lcd_control_wr(LCD_DISPLAY_ON_CURSOR_OFF);
}

/*
 * Send null terminated string s to line l of the LCD
 * Line 'l' = 0,1
 */
void lcd_write(char *s, char l)
{
char a,i;
    a = (l == 0) ? LCD_ADDR_LINE_0 : LCD_ADDR_LINE_1;
    _lcd_control_wr(LCD_SET_DD_RAM_ADDR(a));
    for (i = 0; i < LCD_NUM_CHARS; i++)
    {   if (*s)
            lcd_putc(*s++);
        else
            return;
    }
}

/*
 * Set character at the current character position of the LCD
 */
void lcd_putc(char c)
{
    _lcd_check_busy();                  /* wait til device not busy */
    LCD_CTRL_PORT |= (_BV(LCD_RS));     /* RS -> 1  data register input */
    LCD_DATA_PORT &= ~LCD_DATA_INIT;    /* first set data bits only to zero */
    LCD_DATA_PORT |= (0x0F & (c >> 4)); /* first set the higher nibble */
    _strobe_E();
    LCD_DATA_PORT &= ~LCD_DATA_INIT;    /* first set data bits only to zero */
    LCD_DATA_PORT |= (0x0F & (c));      /* next set the lower nibble */
    _strobe_E();
}

/*
 * Place (seek) character address at position r on line l of the LCD
 */
void lcd_seek(char r, char l)
{
char a;
    a = (l == 0) ? LCD_ADDR_LINE_0 : LCD_ADDR_LINE_1;
    a += (r < LCD_NUM_CHARS) ? r : (LCD_NUM_CHARS-1);
    _lcd_control_wr(LCD_SET_DD_RAM_ADDR(a));
}
